﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

using OTOM;

namespace OTOMUI
{
    public partial class ObjectToObjectMapper : Form
    {
        private bool isDirty = false;

        public ObjectToObjectMapper()
        {
            InitializeComponent();

            if (Environment.GetCommandLineArgs().Length > 0)
            {
                String[] args = Environment.GetCommandLineArgs();

                foreach (String arg in args)
                {
                    if (arg.Contains(OTOMConstants.FILE_EXTENTION))
                    {
                        loadMapping(arg);
                    }
                }                
            }
        }

        private void button1_Click(object sender, EventArgs e)
        {
            if (openAssemblyDialog.ShowDialog() == DialogResult.OK)
            {
                txtAssemblySource.Text = openAssemblyDialog.FileName;
            }
        }

        private void button2_Click(object sender, EventArgs e)
        {
            if (openAssemblyDialog.ShowDialog() == DialogResult.OK)
            {
                txtAssemblyDestination.Text = openAssemblyDialog.FileName;
            }
        }

        private void btnLoad_Click(object sender, EventArgs e)
        {
            lbPairs.Items.Clear();

            if (txtAssemblyDestination.Text.Equals(txtAssemblySource.Text))
            {
                Type[] classes = ClassInfo.getTypesFromAssembly(txtAssemblySource.Text);
                bindListBox(lbClassSource, classes, "Name");
                bindListBox(lbClassDestination, classes, "Name");
            }
            else
            {
                bindListBox(lbClassSource, ClassInfo.getTypesFromAssembly(txtAssemblySource.Text), "Name");
                bindListBox(lbClassDestination, ClassInfo.getTypesFromAssembly(txtAssemblyDestination.Text), "Name");
            }
        }

        private void bindListBox(ListBox listBox, IEnumerable<Type> collection, String displayName)
        {
            listBox.DataSource = collection.OrderBy(p => p.Name).ToList();
            listBox.DisplayMember = displayName;
        }

        private void bindListBox(ListBox listBox, IEnumerable<System.Reflection.PropertyInfo> collection, String displayName)
        {
            listBox.DataSource = collection.OrderBy(p => p.Name).ToList();
            listBox.DisplayMember = displayName;
        }

        private void cbClassSource_SelectedIndexChanged(object sender, EventArgs e)
        {
            bindListBox(lbPropertySource, ((Type)lbClassSource.SelectedItem).GetProperties(), "Name");
        }

        private void cbClassDestination_SelectedIndexChanged(object sender, EventArgs e)
        {
            bindListBox(lbPropertyDestination, ((Type)lbClassDestination.SelectedItem).GetProperties(), "Name");
        }        

        private void loadFieldInfo(ListBox comboBox, object item)
        {
            comboBox.Items.Add(item);
        }

        private void btnAddMapping_Click(object sender, EventArgs e)
        {
            if (lbPropertySource.SelectedItem != null && lbPropertyDestination.SelectedItem != null)
            {
                PropertyPair pair = new PropertyPair((System.Reflection.PropertyInfo)lbPropertySource.SelectedItem, (System.Reflection.PropertyInfo)lbPropertyDestination.SelectedItem);

                lbPairs.Items.Add(pair);
            }

            this.isDirty = true;
        }

        private void btnGenerate_Click(object sender, EventArgs e)
        {
            if (lbPairs.Items.Count > 0)
            {
                List<PropertyPair> pairs = new List<PropertyPair>(lbPairs.Items.Count);

                foreach(object pair in lbPairs.Items)
                {
                    pairs.Add((PropertyPair)pair);
                }

                String mapping = ObjectMapper.map(pairs, (Type)lbClassSource.SelectedItem, (Type)lbClassDestination.SelectedItem, cbIncludeReverseMapping.Checked);

                new CodeMapping(mapping).ShowDialog(this);
            }
            else
            {
                MessageBox.Show(this, "You must specifiy at least one property mapping before you can generate.", "Unable to generate mapping.", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
            }
        }

        private void btnRemoveMapping_Click(object sender, EventArgs e)
        {
            switch (lbPairs.SelectionMode)
            {
                case SelectionMode.One:
                    if (lbPairs.SelectedItem != null)
                    {
                        lbPairs.Items.Remove(lbPairs.SelectedItem);
                    }
                    break;
                case SelectionMode.MultiExtended:
                case SelectionMode.MultiSimple:
                    List<Object> selection = new List<Object>(lbPairs.SelectedItems.Count);

                    foreach (Object o in lbPairs.SelectedItems)
                    {
                        selection.Add(o);
                    }

                    foreach (Object o in selection)
                    {
                        lbPairs.Items.Remove(o);
                    }
                    break;
            }

            this.isDirty = true;
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Application.Exit();
        }

        private void aboutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            new AboutOTOM().Show(this);
        }

        private void btnSaveMapping_Click(object sender, EventArgs e)
        {
            if (lbPairs.Items.Count > 0)
            {
                Mapping mapping = createMapping();

                saveMappingDialog.FileName = ((Type)lbClassSource.SelectedItem).Name + "To" + ((Type)lbClassDestination.SelectedItem).Name + OTOMConstants.FILE_EXTENTION;

                if (saveMappingDialog.ShowDialog(this) == DialogResult.OK)
                {
                    String filename = saveMappingDialog.FileName;

                    saveMapping(mapping, filename);
                }
            }
            else
            {
                MessageBox.Show(this, "You must specifiy at least one property mapping before you can save.","Unable to save mapping.", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
            }
        }

        private void saveMapping(Mapping mapping, String filename)
        {
            mapping.SaveToDisk(filename);
        }

        private Mapping createMapping()
        {
            AssemblyMapping assMapSource = AssemblyMapping.Instance(ClassInfo.getAssemblyByProperty(((PropertyPair)lbPairs.Items[0]).Source));
            AssemblyMapping assMapDest = AssemblyMapping.Instance(ClassInfo.getAssemblyByProperty(((PropertyPair)lbPairs.Items[0]).Destination));

            Mapping mapping = new Mapping();

            mapping.SourceAssembly = assMapSource;
            mapping.DestinationAssembly = assMapDest;

            foreach (PropertyPair pair in lbPairs.Items)
            {
                mapping.PropertyPairs.Add(new PropertyMapping(pair.Source.Name, pair.Destination.Name, pair.Source.ReflectedType.FullName, pair.Destination.ReflectedType.FullName));
            }

            return mapping;
        }

        private void loadToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (openMappingDialog.ShowDialog() == DialogResult.OK)
            {
                String filename = openMappingDialog.FileName;

                loadMapping(filename);
            }
        }

        private void loadMapping(String filename)
        {
            this.isDirty = false;

            Mapping mapping = Mapping.LoadFromDisk(filename);

            txtAssemblyDestination.Text = mapping.DestinationAssembly.Uri;
            txtAssemblySource.Text = mapping.SourceAssembly.Uri;

            btnLoad_Click(null, null);

            lbClassSource.SelectedItem = ClassInfo.getTypeByName(mapping.SourceAssembly.Uri, mapping.PropertyPairs[0].SourceType);
            lbClassDestination.SelectedItem = ClassInfo.getTypeByName(mapping.DestinationAssembly.Uri, mapping.PropertyPairs[0].DestinationType);

            foreach (PropertyMapping propMapping in mapping.PropertyPairs)
            {
                PropertyPair pair = new PropertyPair();

                pair.Source = ClassInfo.getPropertyByName(mapping.SourceAssembly.Uri, propMapping.SourceType, propMapping.SourceName);
                pair.Destination = ClassInfo.getPropertyByName(mapping.DestinationAssembly.Uri, propMapping.DestinationType, propMapping.DestinationName);

                lbPairs.Items.Add(pair);
            }
        }
    }
}